<?php

error_reporting(E_ALL);
ini_set('display_errors', 1);

/**
 * @package    Banwire Platform
 * @author     Roberto I Ramirez <rramirez@banwire.com>
 * @phpver     >= 5.0
 * @copyright  2011 BanWire
 * @version    2.0
 */
Class Oxxo
{

    /**
     * URL del webservice
     *
     * @var string
     */
    private $_host = 'https://www.banwire.com/api.oxxo';

    /**
     * Ruta donde se almacenará la imagen del código de barras.
     *
     * @var string
     */
    private $_path = '/';

    /**
     * Array contenedor de parametros para la API.
     *
     * @var array
     */
    private $_params = array(
        'usuario' => NULL, // Nombre del usuario en BanWire.
        'referencia' => NULL, // Referencia de la transacción.
        'dias_vigencia' => NULL, // Días que tendrá para realizar el pago, máximo 10 días.
        'monto' => NULL, // Monto a cobrar, máximo 10000.
        'url_respuesta' => NULL, // URL a donde se mandara el POST de la respuesta.
        'cliente' => NULL, // Nombre del cliente.
        'formato' => 'HTTPQUERY', // Formato de la comunicación JSON | HTTPQUERY.
        'email' => NULL, // Si sendPDF es TRUE, manda el recibo de pago a este correo.
        'sendPDF' => FALSE // Indica si BanWire envia el mail de pago.
    );

    /**
     * Contenedor de la respuesta, si procesas, de la API.
     *
     * @var string
     */
    private $_buffer = '';

    /**
     * Contenedor de la respuesta, procesada, de la API.
     *
     * @var array
     */
    private $_response = array();

    /**
     * Funcion constructora.
     *
     * @param string $host Dirección de la API.
     * @param string $path Ruta donde se almacenara la imagen del código de barras.
     */
    public function __construct($host = '', $path = NULL)
    {
        if (!empty($host))
            $this->_host = $host;
        if (!is_null($path))
            $this->_path = $path;
    }

    /**
     * Establece parametros de envío.
     *
     * @params string $key Nombre de la variable.
     * @params string $val Valor de la variable.
     */
    public function __set($key = '', $val = '')
    {
        if (empty($key) || empty($val))
            return FALSE;
        if (!array_key_exists($key, $this->_params))
            return FALSE;
        $this->_params[$key] = $val;
    }

    /**
     *   Funcion para acceder el valor de una variable, se
     * debe usar despues de enviar los datos usando send()
     * ya que depende de las variables de respuesta.
     *
     * @params string $key Nombre de la variable retornada
     * @return string
     */
    public function __get($key)
    {
        if (empty($this->_response))
            return FALSE;
        if (!array_key_exists($key, ($this->_response['error'] ? $this->_response : $this->_response['response'])))
            return FALSE;
        return $this->_response['error'] ? $this->_response[$key] : $this->_response['response'][$key];
    }

    /**
     *   Envia las variables al servidor de BanWire mediante
     * metodo POST usando cURL y procesa la respuesta.
     *
     * @params array $postfields Matriz, opcional, de variables a enviar.
     */
    public function send($postfields = array())
    {
        $postfields = empty($params) ? $this->_params : $params;
        if (!function_exists('curl_init'))
            return FALSE;
        if (!function_exists('http_build_query')) { // Crear cadena en formato http query
            foreach ($postfields as $i => $val)
                $pf .= urlencode($i) . '=' . urlencode($val) . "&";
            $pf = substr($pf, 0, strlen($pf) - 1);
        }
        else
            $pf = http_build_query($postfields);
        $postfields = $pf;
        $ch = curl_init(); // Iniciamos conexion cURL    
        curl_setopt($ch, CURLOPT_URL, $this->_host);
        curl_setopt($ch, CURLOPT_USERAGENT, 'Mozilla/4.0 (compatible; MSIE 6.0; WINDOWS; .NET CLR 1.1.4322)');
        curl_setopt($ch, CURLOPT_HTTPHEADER, array('Accept-Charset' => 'utf-8,*'));
        curl_setopt($ch, CURLOPT_MAXREDIRS, 10);
        curl_setopt($ch, CURLOPT_SSL_VERIFYHOST, 0);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, 0);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, 1);
        curl_setopt($ch, CURLOPT_FOLLOWLOCATION, 1);
        curl_setopt($ch, CURLOPT_BINARYTRANSFER, 1);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        curl_setopt($ch, CURLOPT_HEADER, 0);
        curl_setopt($ch, CURLOPT_POST, 1);
        curl_setopt($ch, CURLOPT_POSTFIELDS, $postfields);
        if (defined('CURLOPT_ENCODING'))
            curl_setopt($ch, CURLOPT_ENCODING, "");
        $this->_buffer = curl_exec($ch); // Almacenamos resultado
        if ($this->_buffer == null)
            return FALSE;
        if (!defined('CURLOPT_ENCODING'))
            return FALSE;
        curl_close($ch);
        return $this->_parseResponse();
    }

    /**
     * Procesa la respuesta de la API.
     *
     * @param string $buffer Respuesta de la API.
     */
    private function _parseResponse($buffer = '')
    {
        if (empty($buffer))
            $buffer = $this->_buffer;
        if (empty($buffer))
            return FALSE;
        switch (strtoupper($this->_params['formato'])) {
            case 'JSON':
                $this->_response = json_decode($buffer, TRUE);
                break;
            case 'HTTPQUERY':
            case 'HTTP_QUERY':
                parse_str($buffer, $this->_response);
                break;
            case 'XML':
                break;
            case 'ARRAY':
                break;
        }
        if ($this->_validateTransaction()) { // Transacción valida.
            $this->_generateFile();
            $this->_response['response']['path_img'] = $this->getPathImage();
            return TRUE;
        }
        else // Error en la transacción.
            return FALSE;
    }

    /**
     * Valida que la transaccion haya sido exitosa.
     *
     * @return boolean.
     */
    private function _validateTransaction()
    {
        if (empty($this->_response))
            return FALSE;
        return !$this->_response['error'];
    }

    /**
     * Genera la imagen del codigo.
     */
    private function _generateFile()
    {
        file_put_contents($this->getPathImage(), base64_decode($this->_response['response']['barcode_img']));
        @chmod($this->getPathImage(), 0777);
        return TRUE;
    }

    /**
     * Obtiene la respuesta procesada de la API.
     *
     * @return array
     */
    public function getResponse()
    {
        return $this->_response; //($this->_response['error']) ? $this->_response : $this->_response['response'];
    }

    /**
     * Genera el nombre y la ruta donde sera almacenada la imagen del codigo de barras.
     *
     * @return string
     */
    public function getPathImage()
    {
        return "$this->_path{$this->_params['referencia']}.png";
    }

    public function getError()
    {
        return ($this->_response['error']) ? $this->_response['error_msg'] : FALSE;
    }

}

/**
 * Ejemplo de uso.
 */
$OXXO = new Oxxo('', '/var/www/vhosts/shop6090.com/development/oxxo_PXF5918/imagenes/');
$OXXO->usuario = 'desarrollo';
//$OXXO->referencia = 'Z123X789Z';
$OXXO->referencia = '100000214'; # es la orden de compra, incluso, se usará para generar la imagen del código de barras: 100000213.png
$OXXO->dias_vigencia = 7;
$OXXO->monto = 120;
//$OXXO->url_respuesta = 'http://banwire.com';
$OXXO->url_respuesta = 'http://development.shop6090.com/banwire_UFC3920/oxxo/catch_response.php';
$OXXO->cliente = 'Zarampahuilo Ramírez García';
$OXXO->formato = 'JSON';
/* BanWire enviara el mail con el PDF */
$OXXO->sendPDF = FALSE;
$OXXO->email = 'uramos@gmail.com';
if ($OXXO->send()) {
    // Acciones a realizar si todo salió bien.
} else {
    // Acciones a realizar si surgió un error.
}
